/*=======================================================*/
/*
 *  Listing 5.  userdev.c - user-defined device functions
 *
 *  This file contains 'skeleton' procedures for a
 *  device.  These procedures are called from procedures
 *  in devinit.c.  They are 'fleshed out' to implement
 *  the functionality for an actual device.
 *
 *  Placed in the Public Domain by Forest W. Arnold, 1994
*/
/*=======================================================*/

#include "userdev.h"

/*
 *  define the device name and id string - these are
 *  referenced from the code in the assembly language
 *  device initialization module.  The DEV_NAME and
 *  DEV_IDSTR macros are defined in the userdev.h
 *  include file.
*/

char dev_Name[] = DEV_NAME;
char dev_IdStr[] = DEV_IDSTR;

/*
 *  prototypes for device specific procedures called from 
 *  the procedures in this file
*/

int cmd_Invalid( struct IOStdReq * );
int cmd_Reset( struct IOStdReq * ); 
int cmd_Read( struct IOStdReq * );  
int cmd_Write( struct IOStdReq * ); 
int cmd_Update( struct IOStdReq * );
int cmd_Clear( struct IOStdReq * ); 
int cmd_Stop( struct IOStdReq * );  
int cmd_Start( struct IOStdReq * ); 
int cmd_Flush( struct IOStdReq * ); 
int cmd_Debug( struct IOStdReq * ); 


/*
 *  typedef for function pointer to a device command
 *  handler
*/
 
typedef int (*CmdProc_t)( struct IOStdReq * );

/*
 *  table of function pointers.  These are indexed by
 *  the device's commands.  The standard exec commands
 *  are defined in exec/io.h
*/

CmdProc_t dev_Cmds[] =
{
   cmd_Invalid,   /*  0 - CMD_INVALID  */
   cmd_Reset,     /*  1 - CMD_RESET    */
   cmd_Read,      /*  2 - CMD_READ     */
   cmd_Write,     /*  3 - CMD_WRITE    */
   cmd_Update,    /*  4 - CMD_UPDATE   */
   cmd_Clear,     /*  5 - CMD_CLEAR    */
   cmd_Stop,      /*  6 - CMD_STOP     */
   cmd_Start,     /*  7 - CMD_START    */
   cmd_Flush,     /*  8 - CMD_FLUSH    */
   cmd_Debug,     /*  9 - DEV_DBG_ON   */
   cmd_Debug,     /* 10 - DEV_DBG_OFF  */

   /*  add new command procedures here */
};

/*=======================================================*/
/*
 *  usr_devInit - device initialization function
 *
 *  This procedure initializes device-specific data that
 *  is global to the device and shared by all programs
 *  that access the device.
 *
 *  This procedure is called after the device has been
 *  allocated.  When this procedure returns a value
 *  other than zero, the device is added to the system
 *  device list.
 *  If it returns 0, the device is unloaded.
*/
/*=======================================================*/

int usr_devInit( struct GenericDevice *device )
{

   /*
    *  Allocate a global structure containing information
    *  about this device and set the dev_Data.data
    *  structure member to the allocated structure.
    *  init the device-specific data elements
    *  that are added for this device.  Return 0 if
    *  an error occurred and the device is not to be
    *  added to the system.
   */

   return 1;
}

/*=======================================================*/
/*
 *  usr_devOpen() - open a device
 *
 *  This procedure sets up the device for a calling
 *  process to use.  It is single-threaded by exec 
 *  (via Forbid()-Permit()), so it needs to return
 *  quickly.
 *
 *  This procedure is only needed if the device creates
 *  and/or initializes data for each task that opens
 *  the device, or if the device needs to fill in
 *  any of the fields in the io request.
 *
 *  NOTE:  a call to allocate memory in this routine
 *         can trigger a call to Expunge, so it needs
 *         to fake an open count to prevent this if 
 *         it allocates memory
*/
/*=======================================================*/

void usr_devOpen( long unit,struct IOStdReq *req,
                  ULONG flags, struct GenericDevice *dev)
{
   /*
    *  perform initialization actions for the caller's
    *  'open' request.  Set io_Error to non-zero
    *  value to indicate an error.
   */

   return;
}

/*=======================================================*/
/*
 *  usr_devClose() - close a device
 *
 *  This procedure is called to detach a device from a
 *  process that is using it. 
 *
 *  This procedure is only needed if the device needs
 *  to free any memory or clean up anything that was
 *  created for an opening task in usr_devOpen().  
 *  If the open count for the device reaches zero when
 *  it is closed, the usr_devExpunge() procedure is 
 *  called after this procedure returns.
*/
/*=======================================================*/

void usr_devClose( struct IOStdReq *req,
                   struct GenericDevice *dev )
{
   /*
    *  perform any per task device cleanup before
    *  the calling task releases the device
   */

   return;
}

/*=======================================================*/
/*
 *  usr_devExpunge() - remove a device
 *
 *  The memory allocator calls this procedure when system
 *  is low on memory.  Since it is called by the memory
 *  allocator, it MUST return as quickly as possible.
 *
 *  It is also called when the open count for the
 *  device reaches 0.
 *
 *  If this procedure returns 0, the device will not be
 *  removed from the system.  
 *  If it returns a non-zero value, the device will be
 *  removed and the memory allocated for it will be
 *  freed.
 *
*/
/*=======================================================*/

int usr_devExpunge( struct GenericDevice *dev )
{

   /*
    *  No one is using the device.
    *
    *  Return 1 to allow the device to be removed and
    *  its memory freed.  First, free any memory you
    *  allocated in usr_devInit() and set its pointer(s)
    *  to NULL.  
    *
    *  return zero to indicate that the device
    *  is not to be removed from the system.
   */

   return 1;
}

/*=======================================================*/
/*
 *  usr_devBeginIO() - process a device io command.
 * 
 *  This procedure is called from a client task (one that
 *  opened the device).  
*/
/*=======================================================*/

int usr_devBeginIO( struct IOStdReq *req )
{
   int cmd,ret;

   /*
    *  This procedure is multi-threaded.  If the device
    *  is managing a single global data segment, it will
    *  need to lock out other tasks while it is processing
    *  the device command.  This can be done with
    *  an exec signal semaphore.  The device initialization
    *  code in devinit.c allocates and initializes a
    *  signal semaphore.  To use it to prevent task
    *  conflicts, call 
    *
    *      ObtainSemaphore( dev->dev_Data.lock )
    *
    *  If you lock the device as above, you MUST call
    *
    *     ReleaseSemaphore( dev->dev_Data.lock ) 
    *
    *  before this procedure returns.
    *
    *  If this procedure returns a non-zero value and the
    *  QUICK_IO flag is clear, the user's request will be
    *  returned with ReplyMsg().  If the command can not
    *  be processed immediately, the procedure that 
    *  implements the command should clear the QUICK_IO
    *  flag and return 1.
    *
    *  If zero is returned, or the QUICK_IO flag is set,
    *  the device will simply return to the caller. If 
    *  the procedure that implements the command can
    *  perform the command immediately, it should just
    *  return 0.
    *  
   */

   cmd = req->io_Command;

   if ( cmd < 0 || cmd > USRDEV_LASTCMD )
      cmd = 0;

   if ( ! dev_Cmds[cmd] )
      return 0;

   /*
    *  lock the device if necessary (see above notes) 
    *  and process the command
   */

   ret = (*dev_Cmds[cmd])(req);   

   /*
    *  if device was locked, unlock it and return
   */

   return ret;   
}

/*=======================================================*/
/*
 *  usr_devAbortIO() - abort a device io command
 *
 *  This procedure is multi-threaded.  It is called by
 *  tasks that have (or suspect they have) io requests
 *  that have not been processed by the device.  For
 *  example, a read command could be queued by the device
 *  waiting for a write from another task.  This procedure
 *  removes any io requests that are pending for the
 *  calling task and replies to the caller.
 *
 *  This procedure may need to lock the device before
 *  processing the command.  See notes above in 
 *  usr_devBeginIO()
*/
/*=======================================================*/

void usr_devAbortIO( struct IOStdReq *req )
{
   return;
}

/*=======================================================*/
/*
 * Skeleton procedures for the standard device commands
 * specified for all Amiga devices.  Add code to these
 * to process the commands for a specific device.
*/
/*=======================================================*/

/*=======================================================*/
/*
 * cmd_Invalid() - process invalid device commands
*/
/*=======================================================*/

int cmd_Invalid( struct IOStdReq *req )
{
   return 0;
} 

/*=======================================================*/
/*
 * cmd_Reset() - reset a device
*/
/*=======================================================*/

int cmd_Reset( struct IOStdReq *req )
{
   return 0;
} 

/*=======================================================*/
/*
 * cmd_Read() - read from a device
*/
/*=======================================================*/

int cmd_Read( struct IOStdReq *req )
{
   return 0;
} 

/*=======================================================*/
/*
 * cmd_Write() - write to a device
*/
/*=======================================================*/

int cmd_Write( struct IOStdReq *req )
{
   return 0;
} 

/*=======================================================*/
/*
 * cmd_Update() - perform a device 'update' action
*/
/*=======================================================*/

int cmd_Update( struct IOStdReq *req )
{
   return 0;
} 

/*=======================================================*/
/*
 * cmd_Clear() - 'clear' a device
*/
/*=======================================================*/

int cmd_Clear( struct IOStdReq *req )
{
   return 0;
} 

/*=======================================================*/
/*
 * cmd_Stop() - halt a device
*/
/*=======================================================*/

int cmd_Stop( struct IOStdReq *req )
{
   return 0;
} 

/*=======================================================*/
/*
 * cmd_Start() - start a device
*/
/*=======================================================*/

int cmd_Start( struct IOStdReq *req )
{
   return 0;
} 

/*=======================================================*/
/*
 * cmd_Flush() - flush any pending device actions
*/
/*=======================================================*/

int cmd_Flush( struct IOStdReq *req )
{
   return 0;
} 

/*=======================================================*/
/*
 * cmd_Debug() - turn debug trace on/off
 *  
 * This command is specified for the 'generic' device.
 * This implementation can be used 'as is'.
*/
/*=======================================================*/

int cmd_Debug( struct IOStdReq *req )
{
   struct GenericDevice *dev =
         (struct GenericDevice*)req->io_Device;

   if ( req->io_Command == DEV_DBG_ON )
   {
      if ( dev->dev_Data.dbgFH == 0 )
      {
         /*
          *  lock the device by getting the global
          *  semaphore, then get the debug file
          *  handle
         */

         dev->dev_Data.dbgFH = (BPTR)req->io_Data;
      }
      else
         req->io_Error =  DEVERR_DBGBUSY;
   }
   else
   if ( req->io_Command == DEV_DBG_OFF )
   {
      if ( dev->dev_Data.dbgFH == (BPTR)req->io_Data )
         dev->dev_Data.dbgFH = 0;
      else
         req->io_Error = DEVERR_BADFH;
   }
   return 0;
} 

